<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\CrmClient;
use App\Models\CrmCompanyData;
use App\Models\CrmClientStatus;
use App\Models\Training;
use App\Models\CrmField;
use App\Models\Country;
use App\Models\CrmSource;
use App\Models\Region;
use App\Models\Area;
use App\Models\User;
use App\Models\CrmCompanyTraining;
use App\Models\CrmClientEmployees;
use App\Models\CrmSms;
use App\Models\CrmMeetings;
use App\Models\CrmTask;
use App\Models\Status;
use App\Models\CrmInvoice;
use App\Models\CrmInvoicesItem;
use App\Models\CrmPaymentRecord;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CrmClientsImport;
use App\Imports\CrmCompanyImport;
use Illuminate\Support\Str;
use Carbon\Carbon;
use DB;
use File;
use Image;
use App\Helpers\Helper;
use Illuminate\Support\Facades\Validator;


class CrmController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:crm');
        $this->middleware('permission:crm reports')->only('get_reports');;
    }
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function crmHome()
    {
        $user = User::with('target')->where('id',auth()->user()->id)->first();
        $today = Carbon::today();

        // Get sums for the current month
        $month_target = CrmSms::whereIn('type', ['call', 'meeting'])
            ->where('sender_id', $user->id)
            ->whereYear('created_at', $today->year)
            ->whereMonth('created_at', $today->month)
            ->selectRaw('SUM(CASE WHEN type = "call" THEN 1 ELSE 0 END) AS total_calls, 
                         SUM(CASE WHEN type = "meeting" THEN 1 ELSE 0 END) AS total_meetings')
            ->first();
        
        // Get sums for today
        $today_target = CrmSms::whereIn('type', ['call', 'meeting'])
            ->where('sender_id', $user->id)
            ->whereDate('created_at', $today)
            ->selectRaw('SUM(CASE WHEN type = "call" THEN 1 ELSE 0 END) AS total_calls, 
                         SUM(CASE WHEN type = "meeting" THEN 1 ELSE 0 END) AS total_meetings')
            ->first();
        
        $all_invoices = collect($user->invoices());
        
        // Filter and sum for the current month
        $budget_monthly = $all_invoices->where('created_at', '>=', $today->startOfMonth())
                                              ->where('created_at', '<=', $today->endOfMonth())
                                              ->sum('invoice_value');
        
        // Filter and sum for the current year
        $budget_yearly = $all_invoices->where('created_at', '>=', $today->startOfYear())
                                             ->where('created_at', '<=', $today->endOfYear())
                                             ->sum('invoice_value');
        // Accessing the results
        $numbers=[];
        $numbers['call_monthly'] = $month_target->total_calls ?? 0;
        $numbers['meeting_monthly'] = $month_target->total_meetings ?? 0;
        
        $numbers['call_daily'] = $today_target->total_calls ?? 0;
        $numbers['meeting_daily'] = $today_target->total_meetings ?? 0;
        
        
        $numbers['budget_monthly'] = $budget_monthly ?? 0;
        $numbers['budget_yearly'] = $budget_yearly ?? 0;
        
        
        
        // Group invoices by year and month and sum their values
        $monthlyInvoices = $all_invoices->groupBy(function($invoice) {
            return Carbon::parse($invoice->created_at)->format('Y-m'); // Group by year-month
        });
        
        $monthlyInvoiceValues = [];

        foreach ($monthlyInvoices as $yearMonth => $invoices) {
            [$year, $month] = explode('-', $yearMonth);
            $monthlyInvoiceValues[$year][$month] = $invoices->sum('invoice_value');
        }
        $callMeetingCounts = [];
        
        // Loop through each month of the current year (you can adjust the year if needed)
        
        $date = Carbon::now();
        $firstRecord = CrmSms::where('sender_id', $user->id)->orderBy('created_at', 'asc')->first();
        
        $callMeetingCounts = []; // Initialize the array to store results
        
        
        if ($firstRecord) {
            $startYear = $firstRecord->created_at->year;
            $endYear = $today->year;
        
            // Loop through each year from the earliest record year to the current year
            for ($year = $startYear; $year <= $endYear; $year++) {
                // Loop through each month of the year
                for ($month = 1; $month <= 12; $month++) {
                    // Get counts of calls and meetings for the month
                    $month_target = CrmSms::whereIn('type', ['call', 'meeting'])
                        ->where('sender_id', $user->id)
                        ->whereYear('created_at', $year) 
                        ->whereMonth('created_at', $month)
                        ->selectRaw('DATE(created_at) as date, 
                                     COUNT(CASE WHEN type = "call" THEN 1 END) AS total_calls, 
                                     COUNT(CASE WHEN type = "meeting" THEN 1 END) AS total_meetings')
                        ->groupBy('date') // Group by day if you want daily totals
                        ->get();
        
                    // Calculate monthly totals by summing up each day's counts
                    $total_calls = $month_target->sum('total_calls');
                    $total_meetings = $month_target->sum('total_meetings');
        
                    // Store the results for each month
                    $callMeetingCounts[$year][$month] = [
                        'total_calls' => $total_calls,
                        'total_meetings' => $total_meetings,
                    ];
                }
            }
        }
        $monthlyStatistics  = [];
        
        foreach ($monthlyInvoiceValues as $year => $months) {
            foreach ($months as $month => $invoiceValue) {
                $monthlyStatistics[$year][$month] = [
                    'total_invoice_value' => $invoiceValue,
                    'total_calls' => $callMeetingCounts[$year][$month]['total_calls'] ?? 0,
                    'total_meetings' => $callMeetingCounts[$year][$month]['total_meetings'] ?? 0,
                ];
            }
        }
        
        foreach ($callMeetingCounts as $year => $months) {
            foreach ($months as $month => $counts) {
                if (!isset($monthlyStatistics[$year][$month])) {
                    $monthlyStatistics[$year][$month] = [
                        'total_invoice_value' => 0, // Default to 0 if no invoices
                        'total_calls' => $counts['total_calls'],
                        'total_meetings' => $counts['total_meetings'],
                    ];
                }
            }
        }

        
        return view('admin.crm.home',compact('user','numbers','all_invoices','monthlyStatistics'));
    }
    public function clients(Request $request)
    {
        $clients = CrmClient::with(['training', 'status', 'company', 'source'])
            ->where('type', 'client')
            ->orderBy('created_at', 'desc');
        if ($request->status_id == 19) {
            $clients->where('status_id', 19);
            if(!in_array('Crm Manager', auth()->user()->getRoleNames()->toArray()) && !in_array('admin', auth()->user()->getRoleNames()->toArray())){
                $clients->whereHas('responsible_employee', function ( $query) {
                    $query->where('user_id', auth()->id());
                });
            }
            
            
        }elseif (!in_array('Crm Manager', auth()->user()->getRoleNames()->toArray()) && 
            !in_array('admin', auth()->user()->getRoleNames()->toArray())) {
            $clients->whereHas('responsible_employee', function ( $query) {
                $query->where('user_id', auth()->id());
            });
        }
        
        $clients = $clients->paginate(20);
        if ($request->status_id == 19) {
            return view('admin.crm.client.new_clients', compact('clients'));
        }
    
        return view('admin.crm.client.clients', compact('clients'));
    }
    
    public function client_details(CrmClient $client )
    {
       if($client->responsible_employee){
            if (!in_array('Crm Manager',auth()->user()->getRoleNames()->toArray()) &&
                !in_array('admin', auth()->user()->getRoleNames()->toArray()) &&
                    $client->responsible_employee->user_id!= auth()->id()) {
                abort('404');
            }
       }
        $trainings = Training::where('status',1)->get();
        $client = CrmClient::with(['companyTraining', 'status','messages'])->where('id',$client->id)->first();
        $crm_employees = User::where('type','crm_employee')->where('id','!=',auth()->user()->id)->get();
        $crm_employees_task = User::where('type','crm_employee')->where('id','!=',$client->responsible_employee?->user?->id)->get();
        $statuses = Status::all();
        return view('admin.crm.client.client_details',compact('client','crm_employees','crm_employees_task','statuses','trainings'));
    }
    
    public function store_single_Client(Request $request)
    {
        $client_check = CrmClient::where('email',$request->email)->orWhere('phone',$request->phone)->first();
        if($client_check){
            return response()->json(
                ['status'=>'error','message'=>'رقم التليفون او الايميل لديه حساب بالفعل : '.$client_check->name]);
        }
        $crm_client = CrmClient::create([
            "name"=>$request->name,
            "email"=>$request->email,
            "phone"=>$request->phone,
            "phone2"=>$request->phone2,
            "ssn"=>$request->ssn,
            "whatsapp"=>$request->whatsapp,
            "company_id"=>$request->company_id,
            "status_id"=>$request->status_id,
            "come_from"=>$request->come_from,
            "nationality"=>$request->nationality,
            'country_id'=> $request->country_id,
            'area_id'=> $request->area_id,
            'region_id'=> $request->region_id,
            'qualification'=> $request->qualification,
            "type"=>"client"
            ]);
         foreach($request->training_id as $training_id){
            $training = Training::findOrFail($training_id);
            $training_company = CrmCompanyTraining::create([
            'company_id'=> $crm_client->id,
            'training_id'=> $training_id,
                ]);
                
            CrmInvoice::create([
                'client_id'=>$crm_client->id,
                'invoice_number'=>random_int(100000000, 999999999),
                'crm_company_tranining_id'=> $training_company->id,
                'invoice_value'=>$training->price,
                'training_cost'=>$training->price,
                'duration_days'=>$training->overall_days,
                'invoice_status'=>'in_training'
            ]);
            }
            
        CrmClientEmployees::create([
            'crm_client_id'=>$crm_client->id,
            'user_id' => $request->responsible_employee
            ]);    
        CrmClientStatus::create([
            'crm_client_id'=> $crm_client->id,
            'status_id'=> $request->status_id,
            ]);
        return response()->json(['status'=>'success']);
    }
    
    public function delete_training($crm_company_training){
        CrmCompanyTraining::where('id',$crm_company_training)->delete();
        $invoices = CrmInvoice::where('crm_company_tranining_id',$crm_company_training)->get();
        foreach($invoices as $invoice){
            $this->delete_invoice($invoice);
        }
        return back()->with(['success' => trans('home.your_item_deleted_successfully')]);
    }
      
    
    public function client_update(CrmClient $client,Request $request)
    {
        $client_check = CrmClient::where('id', '!=', $client->id)
            ->where(function ($query) use ($request) {
                $query->where('email', $request->email)
                      ->orWhere('phone', $request->phone);
            })->first();
        if($client_check){
            return back()->with('error','رقم التليفون او الايميل لديه حساب بالفعل : '.$client_check->name);
        }
        $client->update([
            "name"=>$request->name,
            "email"=>$request->email,
            "phone"=>$request->phone,
            "phone2"=>$request->phone2,
            "ssn"=>$request->ssn,
            "whatsapp"=>$request->whatsapp,
            "company_id"=>$request->company_id,
            "status_id"=>$request->status_id,
            "come_from"=>$request->come_from,
            "nationality"=>$request->nationality,
            'country_id'=> $request->country_id,
            'area_id'=> $request->area_id,
            'region_id'=> $request->region_id,
            'qualification'=> $request->qualification,
            'job'=> $request->job,
            "type"=>"client",
            'status_id'=> $request->status_id
            
            ]);
     
        
        if($request->responsible_employee){
            CrmClientEmployees::updateOrCreate(
                ['crm_client_id' => $client->id],
                
                [
                    'crm_client_id' => $client->id,
                    'user_id' => $request->responsible_employee
                ]
            );
        }

        CrmClientStatus::create([
            'crm_client_id'=> $client->id,
            'status_id'=> $request->status_id,
            ]);
        return redirect('admin/crm/clients/'.$client->id)->with('success',trans('home.your_item_updated_successfully'));;
    }
    
    public function addsingleClient( )
    {
        $companies = CrmClient::where('type','company')->get();
        $trainings = Training::all();
        $statuses = Status::all();
        $countries = Country::all();
        $sources = CrmSource::all();
        $crm_employees = User::where('type','crm_employee')->get();
        return view('admin.crm.client.add_client',compact('companies','statuses','trainings','sources','countries','crm_employees'));
    }
    public function addClientCompany( )
    {
        $companies = CrmClient::where('type','company')->get();
        $trainings = Training::all();
        $statuses = Status::all();
        $countries = Country::all();
        $sources = CrmSource::all();
        $crm_employees = User::where('type','crm_employee')->get();
        return view('admin.crm.company.add_client_to_company',compact('companies','statuses','trainings','sources','countries','crm_employees'));
    }
  
    public function client_edit(CrmClient $client )
    {
        if($client->responsible_employee){
            if (!in_array('Crm Manager',auth()->user()->getRoleNames()->toArray()) 
                && $client->responsible_employee->user_id!= auth()->id() 
                &&
                !in_array('admin',auth()->user()->getRoleNames()->toArray())
            ) {
                abort('404');
            }
        }
        $companies = CrmClient::where('type','company')->get();
        $trainings = Training::all();
        $statuses = Status::all();
        $countries = Country::all();
        $regions = Region::all();
        $areas = Area::all();
        $sources = CrmSource::all();
        $crm_employees = User::where('type','crm_employee')->get();
        return view('admin.crm.client.edit_client',compact('regions','areas','client','companies','statuses','trainings','sources','countries','crm_employees'));
    }

    public function companies()
    {
        $clients = CrmClient::with(['companyTraining', 'status', 'company','source','company_data','clients'])
                ->where('type','company')
                ->orderBy('updated_at', 'desc')
                ->paginate(20);
        
        return view('admin.crm.company.companies',compact('clients'));
    }
    
    public function company_details(CrmClient $company )
    {
        $company = CrmClient::with(['companyTraining','company_data', 'status','messages','invoices'])->where('id',$company->id)->first();
        $crm_employees = User::where('type','crm_employee')->where('id','!=',auth()->user()->id)->get();
        $crm_employees_task = User::where('type','crm_employee')->where('id','!=',$company->responsible_employee?->user?->id)->get();
        $statuses = Status::all();
        
        return view('admin.crm.company.company_details',compact('company','crm_employees','crm_employees_task','statuses'));
    }
    
    public function company_edit(CrmClient $company )
    {
        $trainings = Training::all();
        $statuses = Status::all();
        $countries = Country::all();
        $regions = Region::all();
        $areas = Area::all();
        $sources = CrmSource::all();
        $fields = CrmField::all();
        $crm_employees = User::where('type','crm_employee')->get();
        return view('admin.crm.company.edit_company',compact('regions','areas','fields','company','statuses','trainings','sources','countries','crm_employees'));
    }
   
     public function addsingleCompany( )
    {
        $trainings = Training::all();
        $statuses = Status::all();
        $fields = CrmField::all();
        $countries = Country::all();
        $regions = Region::all();
        $sources = CrmSource::all();
        $crm_employees = User::where('type','crm_employee')->get();
        return view('admin.crm.company.add_company',compact('statuses','trainings','fields','countries','crm_employees','sources','regions'));
    }
    
    
    //craete company clients
    public function addCompanyClient(CrmClient $company)
    {
        $trainings = Training::all();
        $statuses = Status::all();
        $countries = Country::all();
        $sources = CrmSource::all();
        $crm_employees = User::where('type','crm_employee')->get();
        return view('admin.crm.company.add_company_client',compact('company','statuses','trainings','sources','countries','crm_employees'));
    }
        
    //store company clients
    public function store_company_Client(Request $request)
    {
       
        $client_check = CrmClient::where('email',$request->email)->orWhere('phone',$request->phone)->first();
        if($client_check){
            return response()->json(
                ['status'=>'error','message'=>'رقم التليفون او الايميل لديه حساب بالفعل : '.$client_check->name]);
        }
        $crm_client = CrmClient::create([
            "name"=>$request->name,
            "email"=>$request->email,
            "phone"=>$request->phone,
            "phone2"=>$request->phone2,
            "ssn"=>$request->ssn,
            "whatsapp"=>$request->whatsapp,
            "company_id"=>$request->company_id,
            "status_id"=>19,
            "come_from"=>$request->come_from,
            "nationality"=>$request->nationality,
            'country_id'=> $request->country_id,
            'area_id'=> $request->area_id,
            'region_id'=> $request->region_id,
            "type"=>"client"
            ]);  
        CrmClientStatus::create([
            'crm_client_id'=> $crm_client->id,
            'status_id'=> 19,
            ]);
        return redirect()->route('company_details',$request->company_id)->with(['success' => trans('home.your_item_added_successfully')]);
    }
    
    public function store_single_Company(Request $request)
    {
        $client_check = CrmClient::where('email',$request->email)->orWhere('phone',$request->phone)->first();
        if($client_check){
            return response()->json(
                ['status'=>'error','message'=>'رقم التليفون او الايميل لديه حساب بالفعل : '.$client_check->name]);
        }
        $data_check = CrmCompanyData::where('company_website',$request->company_website)->first();
        if($data_check){
            return response()->json(
                ['status'=>'error','message'=>'موقع الشركة بالفعل متسجل لشركة : '.$data_check->owner->name]);
        }
         $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string',
            'company_website' => 'nullable|string|max:20',
            'whatsapp' => 'nullable|string|max:20',
            'status_id' => 'required',
            'come_from' => 'required|string|max:255',
            'field_id' => 'nullable|exists:crm_fields,id',
            'country_id' => 'nullable|exists:countries,id',
            'area_id' => 'nullable|exists:areas,id',
            'region_id' => 'nullable|exists:regions,id',
        ]);
    
        // Check if validation fails
        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }
        
         DB::beginTransaction(); // Start the transaction

        try {
            
            $crm_client = CrmClient::create([
                "name"=>$request->name,
                "email"=>$request->email,
                "phone"=>$request->phone,
                "phone2"=>$request->phone2,
                "ssn"=>$request->ssn,
                "whatsapp"=>$request->whatsapp,
                "status_id"=>$request->status_id,
                "come_from"=>$request->come_from,
                "type"=>"company"
            ]);
        
            CrmCompanyData::create([
                'crm_client_id'=> $crm_client->id,
                'responsible_contact_person'=> $request->responsible_contact_person,
                'responsible_contact_phone'=> $request->responsible_contact_phone,
                'responsible_contact_email'=> $request->responsible_contact_email,
                'category'=> $request->category,
                'job'=> $request->job,
                'company_website'=> $request->company_website,
                'field_id'=> $request->field_id,
                'country_id'=> $request->country_id,
                'area_id'=> $request->area_id,
                'region_id'=> $request->region_id,
                ]);
            CrmClientStatus::create([
                'crm_client_id'=> $crm_client->id,
                'status_id'=> $request->status_id,
                ]);
                
                
            if($request->responsible_employee){
                CrmClientEmployees::create(
                    [   'crm_client_id' => $crm_client->id,
                        'user_id' => $request->responsible_employee
                    ]
                );
            }
            DB::commit(); 
            return response()->json(['status'=>'success']);
        }
        catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 500);
        }
    }
    
  
    public function company_update(CrmClient $company,Request $request)
    {
        $client_check = CrmClient::where('id', '!=', $company->id)
            ->where(function ($query) use ($request) {
                $query->where('email', $request->email)
                      ->orWhere('phone', $request->phone);
            })->first();

        if($client_check){
            return back()->with('error','رقم التليفون او الايميل لديه حساب بالفعل : '.$client_check->name);
        }
        $data_check = CrmCompanyData::where('company_website',$request->company_website)->where('crm_client_id','!=',$company->id)->first();
        if($data_check){
            return back()->with('error','موقع الشركة بالفعل متسجل لشركة : '.$data_check->owner->name);
        }
        $company->update([
            "name"=>$request->name,
            "email"=>$request->email,
            "phone"=>$request->phone,
            "ssn"=>$request->ssn,
            "whatsapp"=>$request->whatsapp,
            "come_from"=>$request->come_from,
            'country_id'=> $request->country_id,
            'area_id'=> $request->area_id,
            'region_id'=> $request->region_id
            ]);
            
        // CrmCompanyTraining::where('company_id',$company->id)->whereNotIn('training_id',$request->training_id)->delete();
        
        // foreach ($request->training_id as $training_id) {
        //     CrmCompanyTraining::firstOrCreate(
        //         [
        //             'company_id' => $company->id,
        //             'training_id' => $training_id
        //         ]
        //     );
        // }
        CrmCompanyData::updateOrCreate(
            ['crm_client_id'=>$company->id],
            [
            'responsible_contact_person'=> $request->responsible_contact_person,
            'responsible_contact_phone'=> $request->responsible_contact_phone,
            'responsible_contact_email'=> $request->responsible_contact_email,
            'category'=> $request->category,
            'job'=> $request->job,
            'field_id'=> $request->field_id,
            'country_id'=> $request->country_id,
            'area_id'=> $request->area_id,
            'region_id'=> $request->region_id,
            
        ]);
        if($request->responsible_employee){
            CrmClientEmployees::updateOrCreate(
                ['crm_client_id' => $company->id],
                
                [
                    'crm_client_id' => $company->id,
                    'user_id' => $request->responsible_employee
                ]
            );
        }

        // CrmClientStatus::create([
        //     'crm_client_id'=> $company->id,
        //     'status_id'=> $request->status_id,
        //     ]);
        return redirect('admin/crm/companies/'.$company->id)->with('success',trans('home.your_item_updated_successfully'));
    }
    
    public function send_sms(CrmClient $client ,Request $request){
        if($request->type=="meeting"){
            
            $date = $request->date;
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $start_date_parsed = Carbon::parse($request->start_date);
            $end_date_parsed = Carbon::parse($request->end_date);
            if($end_date_parsed<=$start_date_parsed){
                return back()->with('error', 'End date must be after start date.');
            }
            foreach($request->team as $crm_employee_id){
                $old_meeting = CrmMeetings::whereJsonContains('team_ids',$crm_employee_id)
                    ->where('date',$date)
                     ->where(function($query) use ($start_date, $end_date) {
                        $query->whereBetween('start_date', [$start_date, $end_date])
                              ->orWhereBetween('end_date', [$start_date, $end_date])
                              ->orWhere(function($query) use ($start_date, $end_date) {
                                  $query->where('start_date', '<=', $start_date)
                                        ->where('end_date', '>=', $end_date);
                              });
                    })
                    ->first();
                if($old_meeting){
                    $user = User::findOrfail($crm_employee_id);
                    return back()->with('error', __('home.team').': '.$user->name().__('home.has a meeting at the same time.'));
                }
            }
        }
        if($request->type=="task" && $request->notify_date>$request->date){
            return back()->with('error', __('home.notification date can not be after your date'));
        }
        $message = CrmSms::create([
            'sender_id'=>auth()->user()->id,
            'call_object'=>$request->call_object,
            'result'=>$request->result,
            'message'=>$request->message,
            'client_id'=>$client->id,
            'type'=>$request->type,
            'sent'=>$request->send_to_client,
            'sent_to'=>$request->phone
            ]);
            
        if($request->type=="meeting"){
            CrmMeetings::create([
                "crm_sms_id"=>$message->id,
                "created_by"=>auth()->user()->id,
                "team_ids"=>$request->team,
                "title"=>$request->title,
                "note"=>$request->note,
                "location"=>$request->location,
                "result"=>$request->result,
                "address"=>$request->address,
                "zoom_link"=>$request->zoom_link,
                "date"=>$request->date,
                "start_date"=>$request->start_date,
                "end_date"=>$request->end_date,
                "meeting_type"=>$request->meeting_type
                ]);
        }
            
        if($request->type=="task"){
            CrmTask::create([
                "crm_sms_id"=>$message->id,
                "created_by"=>auth()->user()->id,  // created_by auth user
                "priority"=>$request->priority,
                "task_type"=>$request->task_type,
                "date"=>$request->date,
                "time"=>$request->time,
                "notify_date"=>$request->notify_date,
                "notify_time"=>$request->notify_time,
                "employee"=>$request->employee, // auht one or others according if auth admin or not
                "note"=>$request->note
                ]);
        }
        if($request->send_to_client){
            if($request->type=="sms"){
                // send message to $request->phone
            }
            if($request->type=="whatsapp"){
                // send message whatsapp to $request->phone
            }
            if($request->type=="email"){
                // send message whatsapp to $request->phone
            }
            if($request->type=="meeting"){
                // send message meeting to $request->phone
            }
        }
        if($client->type=="client"){
            return redirect('admin/crm/clients/'.$client->id)->with('success',trans('home.your_item_sent_successfully'));
        }else{
            return redirect('admin/crm/companies/'.$client->id)->with('success',trans('home.your_item_sent_successfully'));
        }
    }
    
    
    public function delete_crm_sms(Request $request){
        CrmSms::where('id',$request->id)->delete();
        return back()->with(['success' => trans('home.your_item_deleted_successfully')]);
    }
    
    public function save_payment(CrmClient $client ,Request $request){
        $request->validate([
            'invoice_status' => 'required',
            'invoice_value' => 'required',
            'training_cost' => 'required',
        ]);
        if($request->training_id){
            $training = Training::findOrFail($request->training_id);
            $training_company = CrmCompanyTraining::create([
            'company_id'=> $client->id,
            'training_id'=> $request->training_id,
                ]);
        }else{
            $training_company=null;
        }
            
        CrmInvoice::create([
            'client_id'=>$client->id,
            'invoice_number'=>random_int(100000000, 999999999),
            'crm_company_tranining_id'=> $training_company?->id,
            'n_clients'=>$request->n_clients,
            'n_trainings'=>$request->n_trainings,
            'invoice_value'=>$request->invoice_value,
            'training_cost'=>$request->training_cost,
            'training_loaction'=>$request->training_loaction,
            'duration_days'=>$request->overall_days,
            'duration_weeks'=>$request->duration_weeks,
            'duration_hours'=>$request->duration_hours,
            'invoice_status'=>$request->invoice_status
        ]);
        return back()->with(['success' => trans('home.your_item_added_successfully')]);
    }
    
    public function invoice_details(CrmInvoice $invoice){
        $items = CrmInvoicesItem::where('crm_invoice_id',$invoice->id)->get();
        return view('admin.crm.company.invoice_details',compact('invoice','items'));
    }
    
    public function invoice_store(Request $request){
        $invoice = CrmInvoice::findOrFail($request->invoice_id);
        $file = Helper::uploadFile('invoices',$request->invoice_file);
        $all_collected = 0 ;
        $items = CrmInvoicesItem::where('crm_invoice_id',$invoice->id)->get();
        if(count($items)>0){
            $all_collected = $all_collected + $items->sum('amount');
        }
        $total_amount = $all_collected+$request->amount;
        if($total_amount > ($invoice->invoice_value - $invoice->discount??0)){
            return back()->with(['error' => 'القيمة المضافة اكبر من قيمة الفاتورة الاساسية']);
        }
        
        $status = $invoice->invoice_value==($total_amount+$invoice->discount??0)?'collected':'invoicing';
        CrmInvoicesItem::create([
            'client_id'=>$invoice->client_id,
            'crm_invoice_id'=>$request->invoice_id ,
            'amount'=>$request->amount,
            'invoice_file'=>$file,
            'invoice_code'=>$request->invoice_number //random_int(100000000, 999999999),
            ]);
        $invoice->invoice_status=$status;
        $invoice->save();
        return back()->with(['success' => trans('home.your_item_added_successfully')]);
    }
    
    public function delete_invoice(CrmInvoice $invoice){
        $items = CrmInvoicesItem::where('crm_invoice_id',$invoice->crm_invoice_id)->get();
        $img_path = base_path() . '/uploads/invoices/';
        foreach($items as $item){
            if ($item->invoice_file) {
                (file_exists(sprintf($img_path . '%s', $item->invoice_file)))?unlink(sprintf($img_path . '%s', $item->invoice_file)):'';
            }
        }
        $invoice->delete();
        return back()->with(['success' => trans('home.your_item_updated_successfully')]);
    }
    
    public function delete_invoice_item(CrmInvoicesItem $invoice){
        $main_invoice = CrmInvoice::findOrFail($invoice->crm_invoice_id);
        $img_path = base_path() . '/uploads/invoices/';
        if ($invoice->invoice_file) {
            (file_exists(sprintf($img_path . '%s', $invoice->invoice_file)))?unlink(sprintf($img_path . '%s', $invoice->invoice_file)):'';
        }
        $main_invoice->invoice_status="invoicing";
        $main_invoice->save();
        $invoice->delete();
        return back()->with(['success' => trans('home.your_item_updated_successfully')]);
    }
    
    public function edit_discount(CrmInvoice $invoice, Request $request){
        if($request->discount>$invoice->invoice_value){
            return back()->with(['error' => 'القيمة المضافة اكبر من قيمة الفاتورة الاساسية']);
        }
        $invoice->update(['discount'=>$request->discount]);
        return back()->with(['success' => trans('home.your_item_updated_successfully')]);
    }
    
    public function my_tasks(){
        $my_tasks = CrmTask::with('crm_sms')->whereJsonContains('employee', (string) auth()->user()->id)->get()->map(function($task){
            $task->type = $task->getTypeAttribute();
            return $task;
        });
        return view('admin.crm.my_tasks',compact('my_tasks'));
    }
    
    public function my_meetings(){
        $my_meetings = CrmSms::with('meeting')->where('type','meeting');
           // Apply condition if the user is not an admin
        if (auth()->user()->type != "admin") {
            $my_meetings->whereHas('meeting', function ($query) {
                $query->whereJsonContains('team_ids', (string) auth()->user()->id);
            });
        }
        $today= today();
        $my_meetings = $my_meetings->get()->pluck('meeting');
        return view('admin.crm.my_meetings',compact('my_meetings','today'));
    }
    
    public function make_task_comlpeted(CrmTask $task){
        $task->update(['status'=>'completed']);
        return back()->with(['success' => trans('home.your_item_updated_successfully')]);
    }
    public function delay_task(Request $request){
        if($request->notify_date>$request->date){
            return back()->with('error', __('home.notification date can not be after your date'));
        }
        $task = CrmTask::findOrFail($request->task_id);
        $task->update([
            'date' => $request->date,
            'time' => $request->time,
            'notify_date' => $request->notify_date,
            'notify_time' => $request->notify_time,
            'status' => 'delayed'
            ]);
        
        return back()->with(['success' => trans('home.your_item_updated_successfully')]);
    }
    
    public function financial(){
        if (!in_array('Crm Manager', auth()->user()->getRoleNames()->toArray()) && 
            !in_array('admin', auth()->user()->getRoleNames()->toArray())) {
                $user = User::findOrFail(auth()->user()->id);
                $invoices = $user->invoices();
            }else{
                $invoices =  CrmInvoice::with('client')->get();
            }
        
        return view('admin.crm.client.financial',compact('invoices'));
    }
    
    public function get_report_options()
    {
         return view('admin.crm.reports.report_options');
    }
    public function get_reports($type,Request $request)
    {
        $responsible_employee = $request->responsible_employee;
        $date = $request->date;
        $new_clients = CrmClient::where('status_id', 19);
        $crm_employees = User::where('type', 'crm_employee');
        $clients_with_multiple_statuses = CrmClient::has('client_statuses', '>', 1);
        $n_calls = CrmSms::where('type', 'call');
        $n_meeting = CrmMeetings::query();
        $n_whatsapp = CrmSms::where('type', 'whatsapp');
        $n_mails = CrmSms::where('type', 'email');
        $n_sms = CrmSms::where('type', 'sms');
        $n_meetings_types = CrmMeetings::selectRaw('meeting_type, COUNT(*) as count');
       
        
        if($type=="clients"){
            // for filter
            $new_clients = $new_clients->where('type','client');
            $clients_with_multiple_statuses = $clients_with_multiple_statuses->where('type','client');
            $n_calls = $n_calls->whereHas('client', function ($query) {
                $query->where('type', 'client');
            });
            $n_meeting = $n_meeting->whereHas('clientSms', function ($query) {
                    $query->whereHas('client', function ($query) {
                            $query->where('type', 'client');
                    });
                });
            $n_whatsapp = $n_whatsapp->whereHas('client', function ($query) {
                $query->where('type', 'client');
            });
            $n_mails = $n_mails->whereHas('client', function ($query) {
                $query->where('type', 'client');
            });
            
            $n_sms = $n_sms->whereHas('client', function ($query) {
                $query->where('type', 'client');
            });
            
            $n_meetings_types = $n_meetings_types->whereHas('clientSms', function ($query) {
                    $query->whereHas('client', function ($query) {
                            $query->where('type', 'client');
                    });
                });
            
        
            // Apply date filter if sent
            if ($date) {
                $new_clients->whereDate('created_at', $date);
                $clients_with_multiple_statuses->whereDate('created_at', $date);
                $n_calls->whereDate('created_at', $date);
                $n_meeting->whereDate('created_at', $date);
                $n_whatsapp->whereDate('created_at', $date);
                $n_mails->whereDate('created_at', $date);
                $n_sms->whereDate('created_at', $date);
                $n_meetings_types ->when($date, function ($query) use ($date) {
                        $query->whereDate('created_at', $date);
                    });
            }
        
            if ($responsible_employee) {
                $crm_client_ids = CrmClientEmployees::where('user_id', $responsible_employee)->pluck('crm_client_id')->toArray();
                $new_clients->whereIn('id', $crm_client_ids);
                $clients_with_multiple_statuses->whereIn('id', $crm_client_ids);
                $n_calls->whereIn('client_id', $crm_client_ids);
        
                $crm_meeting_ids = CrmSms::whereIn('client_id', $crm_client_ids)->where('type', 'meeting')->pluck('id')->toArray();
                $n_meeting->whereIn('crm_sms_id', $crm_meeting_ids);
        
                $n_whatsapp->whereIn('client_id', $crm_client_ids);
                $n_mails->whereIn('client_id', $crm_client_ids);
                $n_sms->whereIn('client_id', $crm_client_ids);
        
                $status_counts = Status::withCount(['clients' => function ($query) use ($crm_client_ids, $date) {
                    $query->whereIn('id', $crm_client_ids);
                    $query->where('type', 'client');
                    if ($date) {
                        $query->whereDate('created_at', $date);
                    }
                }]);
                $n_meetings_types = $n_meetings_types->whereIn('crm_sms_id', $crm_meeting_ids) // Ensures it's filtering by client ID
                    ->groupBy('meeting_type')
                    ->get();
                $today = Carbon::today();
                $user = User::find($responsible_employee);
                $all_invoices = collect($user->invoices());
                $budget_yearly = $all_invoices->where('created_at', '>=', $today->startOfYear())
                                                     ->where('created_at', '<=', $today->endOfYear())
                                                     ->sum('invoice_value');
                // Accessing the results
                $target_numbers['required']=$user->target?->budget_yearly;
                $target_numbers['achieved']=$budget_yearly;
            } 
            
            else {
                $status_counts = Status::withCount(['clients' => function ($query) use ($date) {
                    $query->where('type', 'client');
                    if ($date) {
                        $query->whereDate('created_at', $date);
                    }
                }]);
                $target_numbers=[];
        
                $n_meetings_types = $n_meetings_types->groupBy('meeting_type')->get();
            }
            
        }else{
            $new_clients = $new_clients->where('type','company');
            $clients_with_multiple_statuses = $clients_with_multiple_statuses->where('type','company');
            $n_calls = $n_calls->whereHas('client', function ($query) {
                $query->where('type', 'company');
            });
            $n_meeting = $n_meeting->whereHas('clientSms', function ($query) {
                    $query->whereHas('client', function ($query) {
                            $query->where('type', 'company');
                    });
                });
            $n_whatsapp = $n_whatsapp->whereHas('client', function ($query) {
                $query->where('type', 'company');
            });
            $n_mails = $n_mails->whereHas('client', function ($query) {
                $query->where('type', 'company');
            });
            $n_sms = $n_sms->whereHas('client', function ($query) {
                $query->where('type', 'company');
            });
            
            $n_meetings_types = $n_meetings_types->whereHas('clientSms', function ($query) {
                    $query->whereHas('client', function ($query) {
                            $query->where('type', 'company');
                    });
                });
            
            if ($responsible_employee) {
                $crm_client_ids = CrmClientEmployees::where('user_id', $responsible_employee)->pluck('crm_client_id')->toArray();
                $new_clients->whereIn('id', $crm_client_ids);
                $clients_with_multiple_statuses->whereIn('id', $crm_client_ids);
                $n_calls->whereIn('client_id', $crm_client_ids);
        
                $crm_meeting_ids = CrmSms::whereIn('client_id', $crm_client_ids)->where('type', 'meeting')->pluck('id')->toArray();
                $n_meeting->whereIn('crm_sms_id', $crm_meeting_ids);
        
                $n_whatsapp->whereIn('client_id', $crm_client_ids);
                $n_mails->whereIn('client_id', $crm_client_ids);
                $n_sms->whereIn('client_id', $crm_client_ids);
        
                $status_counts = Status::withCount(['clients' => function ($query) use ($crm_client_ids, $date) {
                    $query->whereIn('id', $crm_client_ids);
                    $query->where('type', 'company');
                    if ($date) {
                        $query->whereDate('created_at', $date);
                    }
                }]);
                $n_meetings_types = $n_meetings_types->whereIn('crm_sms_id', $crm_meeting_ids) // Ensures it's filtering by client ID
                    ->groupBy('meeting_type')
                    ->get();
                $today = Carbon::today();
                $user = User::find($responsible_employee);
                $all_invoices = collect($user->invoices());
                $budget_yearly = $all_invoices->where('created_at', '>=', $today->startOfYear())
                                                     ->where('created_at', '<=', $today->endOfYear())
                                                     ->sum('invoice_value');
                // Accessing the results
                $target_numbers['required']=$user->target?->budget_yearly;
                $target_numbers['achieved']=$budget_yearly;
            } 
            else {
                $status_counts = Status::withCount(['clients' => function ($query) use ($date) {
                    $query->where('type', 'company');
                    if ($date) {
                        $query->whereDate('created_at', $date);
                    }
                }]);
                $target_numbers=[];
        
                $n_meetings_types = $n_meetings_types->groupBy('meeting_type')->get();
            }
            
        }
        
        $crm_employees=$crm_employees->get();
        $new_clients = $new_clients->count();
        $clients_with_multiple_statuses = $clients_with_multiple_statuses->count();
        $n_calls = $n_calls->get();
        $n_meeting = $n_meeting->count();
        $n_whatsapp = $n_whatsapp->count();
        $n_mails = $n_mails->count();
        $n_sms = $n_sms->count();
        $status_counts = $status_counts->get()->where('clients_count', '>', 0);
        
        
    
        return view('admin.crm.reports.reports', compact('type','target_numbers',
            'crm_employees', 'new_clients', 'clients_with_multiple_statuses', 'n_calls', 'n_meeting', 'n_whatsapp', 'n_mails', 'n_sms', 'status_counts', 'n_meetings_types'
        ));
    }

    
    public function get_report_detials(Request $request){
        $lang=\LaravelLocalization::getCurrentLocale();
        $date = $request->date;
        $responsible_employee = $request->responsible_employee;
        if($responsible_employee){
            $crm_client_ids = CrmClientEmployees::where('user_id', $responsible_employee)->pluck('crm_client_id')->toArray();
        }
        switch($request->type){
            case "status_changed":
                
                $clients = CrmClient::with(['training', 'status', 'company', 'source'])
                            ->whereHas('client_statuses', function($query) {
                                $query->havingRaw('COUNT(*) > 1');
                            });
                if($responsible_employee){
                    $clients->whereIn('id',$crm_client_ids);
                }
                if($date){
                    $clients->whereDate('created_at', $date);;
                }
                $clients = $clients->get();
                            
                $status = "status_changed";
                return view('admin.crm.reports.reports_details',compact('clients','status'));
                
            break;
            case "n_calls":
                $calls = CrmSms::with('client')->where('type', 'call');
                if($responsible_employee){
                    $calls->whereIn('client_id',$crm_client_ids);
                }
                if($date){
                    $calls->whereDate('created_at', $date);;
                }
                $calls = $calls->get();
                $status = __("home.n_calls");
                return view('admin.crm.reports.calls',compact('calls','status'));
            break;
            case "n_meeting":
                $meetings = CrmMeetings::with('clientSms');
                if($responsible_employee){
                    $crm_meeting_ids = CrmSms::whereIn('client_id', $crm_client_ids)->where('type', 'meeting')->pluck('id')->toArray();
                    $meetings->whereIn('crm_sms_id',$crm_meeting_ids);
                }
                if($date){
                    $meetings->whereDate('created_at', $date);;
                }
                $meetings = $meetings->get();
                $status = "n_meeting";
                return view('admin.crm.reports.meetings',compact('meetings','status'));
            break;
            case "n_whatsapp":
                $whatsapps = CrmSms::with('client')->where('type', 'whatsapp')->get();
                $status = "n_whatsapp";
                return view('admin.crm.reports.whatsapp',compact('whatsapps','status'));
            break;
            case "n_mails":
                $n_mails = CrmSms::with('client')->where('type', 'email')->get();
                $status = "n_mails";
                return view('admin.crm.reports.mails',compact('n_mails','status'));
                
            break;
            case "n_sms":
                $n_sms = CrmSms::with('client')->where('type', 'sms')->get();
                $status = "n_sms";
                return view('admin.crm.reports.sms',compact('n_sms','status'));
            break;
            case "answered":
                $calls = CrmSms::with('client')->where('type', 'call')->where('sent','answered')->get();
                $status = __('home.call').' '.__('home.answered');
                return view('admin.crm.reports.calls',compact('calls','status'));
            break;
            case "not_answered":
                $calls = CrmSms::with('client')->where('type', 'call')->where('sent','not_answered')->get();
                $status = __('home.call').' '.__('home.didnt_answered');
                return view('admin.crm.reports.calls',compact('calls','status'));
            break;
            case "meeting_company_site":
                $meetings = CrmMeetings::with('clientSms')->where('meeting_type','company_site')->get();
                $status = "n_meeting";
                return view('admin.crm.reports.meetings',compact('meetings','status'));
            break;
            case "meeting_client_site":
                $meetings = CrmMeetings::with('clientSms')->where('meeting_type','client_site')->get();
                $status = "n_meeting";
                return view('admin.crm.reports.meetings',compact('meetings','status'));
            break;
            case "status_data":
                $status = Status::with('clients')->where('id',$request->status_id)->first();
                $client_ids = $status->clients->pluck('id')->toArray();
                $clients = CrmClient::with(['training', 'status', 'company', 'source'])->whereIn('id', $client_ids)->get();
                $status = $status->{'name_'.$lang};
                return view('admin.crm.reports.status',compact('clients','status'));
            break;
            
        default:
            return abort(404);
        }
        
    }
    
    public function save_meeting_result(Request $request){
        $meeting = CrmMeetings::findOrFail($request->meeting_id);
        $meeting->update(['result'=>$request->result]);
        return back()->with(['success' => trans('home.your_item_updated_successfully')]);
    }
    
    public function import_clients(Request $request){
        $request->validate([
            'clients_file' => 'required|mimes:xlsx,csv,pdf',
        ]);
        $file = $request->file('clients_file');
    
        // Import the file using Maatwebsite Excel
        Excel::import(new CrmClientsImport(), $file);
        return redirect('admin/crm/clients?status_id=19')->with('success', 'Data uploaded successfully.');
    }
    public function import_companies(Request $request){
        $request->validate([
            'company_file' => 'required|mimes:xlsx,csv,pdf',
        ]);
        $file = $request->file('company_file');
    
        Excel::import(new CrmCompanyImport(), $file);
        return redirect('admin/crm/companies')->with('success', 'Products uploaded successfully.');
    }
    
    
    public function add_employee_task(){
        $crm_employees = User::where('type', 'crm_employee')->get();
        return view('admin.crm.admin_tasks.add_employee_task',compact('crm_employees'));
    }
    public function store_employee_task(Request $request){
        $message = CrmSms::create([
            'sender_id'=>auth()->user()->id,
            'call_object'=>$request->call_object,
            'result'=>$request->result,
            'message'=>$request->message,
            'type'=>$request->type,
            'sent'=>$request->send_to_client,
            'sent_to'=>$request->phone
            ]);
        if($request->type=="task" && $request->notify_date>$request->date){
            return back()->with('error', __('home.notification date can not be after your date'));
        }
        CrmTask::create([
            "crm_sms_id"=>$message->id,
            "created_by"=>auth()->user()->id,  // created_by auth user
            "priority"=>$request->priority,
            "task_type"=>$request->task_type,
            "date"=>$request->date,
            "time"=>$request->time,
            "notify_date"=>$request->notify_date,
            "notify_time"=>$request->notify_time,
            "employee"=>$request->employee,
            "note"=>$request->note
            ]);
        return redirect('admin/crm/employee-tasks')->with('success', __('home.your_item_added_successfully'));
    }
    public function employee_tasks(){
        $tasks = CrmTask::where('created_by',auth()->user()->id)->get();
        return view('admin.crm.admin_tasks.employee_tasks',compact('tasks'));
    }
    
    public function client_delete($ids){
        $ids = explode(',', $ids);
        foreach ($ids as $id) {
            $client = CrmClient::findOrFail($id);
            $client->delete();
        }
    }
    
    

}
