<?php

namespace App\DataTables;

use App\Models\TrainingCourse;
use App\Models\Training;
use App\Models\Region;
use App\Models\User;
use App\Models\Student;
use App\Models\InstructorInfo;
use App\Models\Partner;
use App\Models\TrainingSchadual;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;

class StudentsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->editColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $url = url('admin/students/'.$row->id.'/edit');
                            if(Auth::user()->can('editStudent')){
                            return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('certificate', function($row){
                            $route = route('student_generate_certificates');
                            if(Auth::user()->can('editStudent')){
                            return '<div class="row m-auto text-center">
                                        <form id="view_certificate" method="POST" action="'. $route .'" >
                                            @csrf
                                            <input type="hidden" id="user_ids" value="{{$row->id}}" name="ids">
                                            <input type="hidden" value="{{$row->id}}" name="training_id">
                                            <button class="btn btn-outline-success m-4 col-lg-6 col-sm-12" type="submit">'. __('home.extract_certificate').'</button>
                                        </form>
                                    </div>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('image', function($row){
                        if($row->image){
                            $url = url('uploads/students/'.$row->image);
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }else{
                            $url = url("resources/assets/back/img/noimage.png");
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }
                    })
             ->addColumn('ssn', function($row){
                        return $row->student_info->ssn ?? '-';
             })
            //  ->addColumn('f_name', function($row){
            //         return $row->name();
            //  })
             ->addColumn('mobile', function($row){
                        return $row->student_info->mobile ?? '-';
             })
             ->addColumn('trainings', function($row){
                       return '<button type="button" 
                                    class="btn btn-primary viewtrainings"
                                    data-toggle="modal" onclick="viewModel2('.$row->id.')" data-target="#view_trainingModal_' . $row->id . '" data-std_id="' . $row->id . '">
                                    <i class="fas fa-user-friends fa-xl"></i>
                                </button>';
             })
             ->addColumn('n_training', function($row){
                        return count($row->trainings);
             })
             ->addColumn('course', function($row){
                        $training = Student::where('user_id',$row->id)->first();
                        if($training){
                            $training = Training::find($training->training_id);
                            return app()->getLocale() == 'en' ? $training->name_en : $training->name_ar;
                        } else {
                            return '';
                        }
             })
             ->addColumn('region', function($row){
                        $region_id = $row->student_info->address->region_id ?? null;
                        $region = Region::find($region_id);
                        return app()->getLocale() == 'en' ? $region->name_en ?? '' : $region->name_ar ?? '';
             })
             ->editColumn('status', function($row){
                        return ($row->status == 1)?trans('home.yes'):trans('home.no');
             })
             ->filterColumn('f_name', function($query, $keyword) {
                    $sql = "f_name like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('ssn', function($query, $keyword) {
                    $sql = "students_info.ssn like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('mobile', function($query, $keyword) {
                    $sql = "students_info.mobile like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('course', function($query, $keyword) {
                    $sql = "trainings.name_ar like ? or trainings.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
             ->filterColumn('region', function($query, $keyword) {
                    $sql = "regions.name_ar like ? or regions.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->rawColumns(['trainings','image','action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(User $model): QueryBuilder
    {
        // return $model->newQuery()->select('users.id as id','users.f_name as f_name','users.email as email','si.ssn as ssn','si.mobile as mobile','str.*')->join('students_info as si','si.user_id','=','users.id')->join('student_training_registered as str','str.user_id','=','users.id');
        if(auth()->user()->type=="partner"){
            $center_ids = Partner::where('user_id',auth()->user()->id)->pluck('center_id')->toArray();
            // $center_ids = auth()->user()->partner->centers->pluck('id')->toArray();
            $training_ids = TrainingSchadual::whereIn('center_id',$center_ids)->pluck('training_id')->toArray();
            $student_ids = Student::whereIn('training_id',$training_ids)->pluck('user_id')->toArray();
            return $model
            ->leftJoin('student_training_registered','student_training_registered.user_id','users.id')
            ->leftJoin('trainings','trainings.id','student_training_registered.training_id')
            ->leftJoin('students_info','students_info.user_id','users.id')
            ->leftJoin('student_addresses','student_addresses.student_id','students_info.id')
            ->leftJoin('regions','regions.id','student_addresses.region_id')
            ->select('users.*','students_info.ssn','trainings.name_ar as training_name_ar','trainings.name_en as training_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
            ->with(['student_info','trainings'])
            ->where('type','student')
            ->whereIn('users.id',$student_ids)->newQuery();
        }
        if(auth()->user()->type=="instructor"){
            // $instructor_id = InstructorInfo::where('user_id',auth()->user()->id)->first()->id;
            $training_ids = TrainingCourse::where('instructor_id',auth()->user()->id)->pluck('training_id')->toArray();
            $student_ids = Student::whereIn('training_id',$training_ids)->pluck('user_id')->toArray();
            return $model
            ->leftJoin('student_training_registered','student_training_registered.user_id','users.id')
            ->leftJoin('trainings','trainings.id','student_training_registered.training_id')
            ->leftJoin('students_info','students_info.user_id','users.id')
            ->leftJoin('student_addresses','student_addresses.student_id','students_info.id')
            ->leftJoin('regions','regions.id','student_addresses.region_id')
            ->select('users.*','students_info.ssn','trainings.name_ar as training_name_ar','trainings.name_en as training_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
            ->with(['student_info','trainings'])->where('type','student')
            ->whereIn('users.id',$student_ids)->newQuery();
        }
        if(auth()->user()->type=="coordinator"){
            $training_ids = Training::where('coordinator_id',auth()->user()->id)->pluck('id')->toArray();
            $student_ids = Student::whereIn('training_id',$training_ids)->pluck('user_id')->toArray();
            return $model
            ->leftJoin('student_training_registered','student_training_registered.user_id','users.id')
            ->leftJoin('trainings','trainings.id','student_training_registered.training_id')
            ->leftJoin('students_info','students_info.user_id','users.id')
            ->leftJoin('student_addresses','student_addresses.student_id','students_info.id')
            ->leftJoin('regions','regions.id','student_addresses.region_id')
            ->select('users.*','students_info.ssn','trainings.name_ar as training_name_ar','trainings.name_en as training_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
            ->with(['student_info','trainings'])
            ->where('type','student')->whereIn('users.id',$student_ids)->newQuery();
        }
        if(auth()->user()->type=="marketer"){
            $training_ids = Training::where('marketer_id',auth()->user()->id)->pluck('id')->toArray();
            $student_ids = Student::whereIn('training_id',$training_ids)->pluck('user_id')->toArray();
            return $model
            ->leftJoin('student_training_registered','student_training_registered.user_id','users.id')
            ->leftJoin('trainings','trainings.id','student_training_registered.training_id')
            ->leftJoin('students_info','students_info.user_id','users.id')
            ->leftJoin('student_addresses','student_addresses.student_id','students_info.id')
            ->leftJoin('regions','regions.id','student_addresses.region_id')
            ->select('users.*','students_info.ssn','trainings.name_ar as training_name_ar','trainings.name_en as training_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
            ->with(['student_info','trainings'])
            ->where('type','student')
            ->whereIn('users.id',$student_ids)->newQuery();
        }else {
            return $model
            ->leftJoin('student_training_registered','student_training_registered.user_id','users.id')
            ->leftJoin('trainings','trainings.id','student_training_registered.training_id')
            ->leftJoin('students_info','students_info.user_id','users.id')
            ->leftJoin('student_addresses','student_addresses.student_id','students_info.id')
            ->leftJoin('regions','regions.id','student_addresses.region_id')
            ->select('users.*','students_info.ssn','trainings.name_ar as training_name_ar','trainings.name_en as training_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
            ->with(['student_info','trainings'])
            ->where('type','student')->newQuery();
        }
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('students-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                            ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [ 
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('id')
                    ->title(__('home.id')),
            Column::make('ssn')
                    ->title(__('home.ssn')),
            Column::make('f_name')
                    ->title(__('home.name')),
            Column::make('mobile')
                    ->title(__('home.phone')),
            Column::make('course')
                    ->title(__('home.course')),
            Column::make('region')
                    ->title(__('home.region')),
            Column::make('trainings')
                    ->title(__('home.trainings')),
            Column::make('action')  
                    ->title(__('home.edit')),
            
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Students_' . date('YmdHis');
    }
}
