<?php

namespace App\DataTables;

use App\Models\TrainingCourse;
use App\Models\TrainingSchadual;
use App\Models\Training;
use App\Models\Center;
use App\Models\User;
use App\Models\Instructor;
use App\Models\Partner;
use App\Models\Student;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;
use DB;

class InstructorsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $url = url('admin/instructors/'.$row->id.'/edit');
                            if(Auth::user()->can('editInstructor')){
                            return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('image', function($row){
                        if($row->image){
                            $url = url('uploads/instructors/'.$row->image);
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }else{
                            $url = url("resources/assets/back/img/noimage.png");
                            return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
                        }
                    })
             ->addColumn('ssn', function($row){
                        return $row->instructor_info->ssn ?? '-';
             })
            //  ->addColumn('f_name', function($row){
            //         return $row->name();
            //  })
             ->addColumn('mobile', function($row){
                        return $row->instructor_info->mobile ?? '-';
             })
             ->addColumn('job_title', function($row){
                        return $row->instructor_info->job_title ?? '-';
             })
             ->addColumn('center', function($row){
                    $partner = $row->centers->first();
                    if($partner) {
                        $center = Center::findOrFail($partner->center_id);
                        if(app()->getLocale() == 'ar') return $center->name_ar; else return $center->name_en;
                    } else {
                        return '';
                    }
             })
             ->addColumn('n_training', function($row){
                        return count($row->trainings);
             })
             ->addColumn('status', function($row){
                        return ($row->status == 1)?trans('home.yes'):trans('home.no');
             })
             ->filterColumn('f_name', function($query, $keyword) {
                    $sql = "f_name like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('mobile', function($query, $keyword) {
                    $sql = "instructors_info.mobile like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('email', function($query, $keyword) {
                    $sql = "email like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('ssn', function($query, $keyword) {
                    $sql = "ssn like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('job_title', function($query, $keyword) {
                    $sql = "instructors_info.job_title like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
             ->filterColumn('status', function($query, $keyword) {
                    if ($keyword == 'yes' || $keyword == '%yes%' || $keyword == 'نعم' || $keyword == '%نعم%') {
                        $query->where('users.status', 1);
                    } elseif ($keyword == 'no' || $keyword == '%no%' || $keyword == 'لا' || $keyword == '%لا%') {
                        $query->where('users.status', 0);
                    }
            })
            ->rawColumns(['image','action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(User $model): QueryBuilder
    {
        if(auth()->user()->type=="partner"){
            $center_ids = Partner::where('user_id',auth()->user()->id)->pluck('center_id')->toArray();
            // $center_ids = auth()->user()->partner->centers->pluck('id')->toArray();
            $instructor_ids = Instructor::whereIn('center_id',$center_ids)->pluck('user_id')->toArray();
            return $model
            ->join('instructors_info','instructors_info.user_id','users.id')
            ->select('users.*','instructors_info.ssn','instructors_info.mobile')
            ->with(['instructor_info','trainings'])->where('type','instructor')->whereIn('users.id',$instructor_ids)->newQuery();
        }
        if(auth()->user()->type=="coordinator"){
            $training_ids = Training::where('coordinator_id',auth()->user()->id)->pluck('id')->toArray();
            $instructor_ids = TrainingCourse::whereIn('training_id',$training_ids)->pluck('instructor_id')->toArray();
            if($this->center_id && $this->center_id!="all"){
                return $model
                ->join('instructors_info','instructors_info.user_id','users.id')
                ->join('instructors_centers','instructors_centers.user_id',"users.id")
                ->join('centers','centers.id',"instructors_centers.center_id")
                ->select('users.*','instructors_info.ssn','instructors_info.mobile')
                ->with(['student_info','trainings'])->where('type','instructor')->where('instructors_centers.center_id',$this->center_id)
                ->whereIn('users.id',$instructor_ids)->newQuery();
            }
            return $model
            ->join('instructors_info','instructors_info.user_id','users.id')
            ->select('users.*','instructors_info.ssn','instructors_info.mobile')
            ->with(['student_info','trainings'])->where('type','instructor')->whereIn('users.id',$instructor_ids)->newQuery();
        }
        if(auth()->user()->type=="marketer"){
            $training_ids = Training::where('marketer_id',auth()->user()->id)->pluck('id')->toArray();
            $instructor_ids = TrainingCourse::whereIn('training_id',$training_ids)->pluck('instructor_id')->toArray();
            if($this->center_id && $this->center_id!="all"){
                return $model
                ->join('instructors_info','instructors_info.user_id','users.id')
                ->join('instructors_centers','instructors_centers.user_id',"users.id")
                ->join('centers','centers.id',"instructors_centers.center_id")
                ->select('users.*','instructors_info.ssn','instructors_info.mobile as ins_mobile')
                ->with(['student_info','trainings'])->where('type','instructor')->where('instructors_centers.center_id',$this->center_id)
                ->whereIn('users.id',$instructor_ids)->newQuery();
            }
            return $model
            ->join('instructors_info','instructors_info.user_id','users.id')
            ->select('users.*','instructors_info.ssn','instructors_info.mobile as ins_mobile')
            ->with(['student_info','trainings'])->where('type','instructor')->whereIn('users.id',$instructor_ids)->newQuery();
        }
        else {
            if($this->center_id && $this->center_id!="all"){
                return $model
                ->join('instructors_info','instructors_info.user_id','users.id')
                ->join('instructors_centers','instructors_centers.user_id',"users.id")
                ->join('centers','centers.id',"instructors_centers.center_id")
                ->select('users.*','instructors_info.ssn as ssn','instructors_info.mobile as mobile')
                ->with(['student_info','trainings'])->where('type','instructor')->where('instructors_centers.center_id',$this->center_id)->newQuery();
            }
            return $model
            ->join('instructors_info','instructors_info.user_id','users.id')
            ->select('users.*','instructors_info.ssn as ssn','instructors_info.mobile as mobile')
            ->with(['student_info','trainings'])->where('type','instructor')->newQuery();
        }
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('instructors-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('id')
                    ->title(__('home.id')),
            Column::make('ssn')
                    ->title(__('home.ssn')),
             Column::make('f_name')
                    ->title(__('home.name')),
             Column::make('center')
                    ->title(__('home.partner')),
             Column::make('job_title')
                    ->title(__('home.job_title')),
            // Column::make('email')
            //         ->title(__('home.email')),
            Column::make('mobile')
                    ->title(__('home.phone')),
            Column::make('n_training')
                    ->title(__('home.n_training')),
            Column::make('action')
                    ->title(__('home.edit')),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Instructors_' . date('YmdHis');
    }
}
