<?php

namespace App\DataTables;

use App\Models\TrainingCourse;
use App\Models\Training;
use App\Models\User;
use App\Models\Center;
use App\Models\Student;
use App\Models\InstructorInfo;
use App\Models\Partner;
use App\Models\TrainingSchadual;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;

class CenterSchedualDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
             ->addColumn('mobile', function($row){
                        return $row->student_info->mobile ?? '-';
             })

             ->addColumn('trainings', function($row){
                       return '<button type="button" 
                                    class="btn btn-primary viewtrainings"
                                    data-toggle="modal" onclick="viewModel2('.$row->id.')" data-target="#view_trainingModal_' . $row->id . '" data-std_id="' . $row->id . '">
                                    <i class="fas fa-user-friends fa-xl"></i>
                                </button>';
             })
            ->rawColumns(['trainings','action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */

    public function query(Center $model): QueryBuilder
    {
       if(auth()->user()->type=="partner"){
            $center_ids = Partner::where('user_id',auth()->user()->id)->pluck('center_id')->toArray();
            return $model->whereIn('centers.id',$center_ids)->newQuery();
        }
        else if(auth()->user()->type=="instructor"){
            // $instructor_id = InstructorInfo::where('user_id',auth()->user()->id)->first()->id;
            $center_ids = TrainingCenter::where('instructor_id',auth()->user()->id)->pluck('center_id')->toArray();
            return $model->whereIn('centers.id',$center_ids)->newQuery();
        }
        else if(auth()->user()->type=="coordinator"){
            $training_ids = Training::where('coordinator_id',auth()->user()->id)->pluck('id')->toArray();
            $center_ids = TrainingCenter::whereIn('training_id',$training_ids)->pluck('center_id')->toArray();
            return $model->whereIn('centers.id',$center_ids)->newQuery();
        }
        else if(auth()->user()->type=="marketer"){
            $training_ids = Training::where('marketer_id',auth()->user()->id)->pluck('id')->toArray();
            $center_ids = TrainingCenter::whereIn('training_id',$training_ids)->pluck('center_id')->toArray();
            return $model->whereIn('centers.id',$center_ids)->newQuery();
        }else {
            return $model->newQuery();
        }

    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('students-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc') 
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [ 
            Column::make('code')
                    ->title(__('home.code')),
            Column::make('name_en')
                    ->title(__('home.name_en')),
            Column::make('name_ar')
                    ->title(__('home.name_ar')),
            Column::make('n_trainings')
                    ->title(__('home.n_trainings')),
            Column::make('date_from')
                    ->title(__('home.date_from')),
            Column::make('date_to')
                    ->title(__('home.date_to')),
            Column::make('trainings')
                    ->title(__('home.trainings')),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Students_' . date('YmdHis');
    }
}
