<?php

namespace App\DataTables;

use App\Models\TrainingCenter;
use App\Models\Training;
use App\Models\Partner;
use App\Models\User;
use App\Models\CenterResponsible;
use App\Models\InstructorInfo;
use App\Models\Center;
use App\Models\CenterHall;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Auth;

class CenterHallDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('checkbox', function($row){
                     return '<input type="checkbox" name="checkbox" class="tableChecked" value="'. $row->id .'" />';
                    })
        
            ->addColumn('action', function($row){
                            $url = url('admin/center_halls/'.$row->id.'/edit');
                            if(Auth::user()->can('editCenterHall')){
                            return '<a href="' . $url . '" class="btn btn-primary" ><i class="fas fa-edit"></i></a>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('map_url', function($row){
                            if($row->map_url){
                            return '<a target="_blank" href="' . $row->map_url . '" class="btn btn-primary" ><i class="fas fa-map-marker-alt"></i></a>';
                            }else {
                                return '';
                            }
                    })
            ->addColumn('address', function($row){
                        if(app()->getLocale() == 'en') $address = $row->address_en ?? '';
                        else $address = $row->address_ar ?? '';
                        return $address;
                    })
            ->addColumn('center_name', function ($row) {
                if(app()->getLocale() == 'en') $center = $row->center->name_en ?? '';
                else $center = $row->center->name_ar ?? '';
                return $center;
            })
            ->addColumn('region_name', function ($row) {
                if(app()->getLocale() == 'en') $region = $row->region->name_en ??'';
                else $region = $row->region->name_ar ?? '';
                return $region;
            })
            ->filterColumn('name_ar', function($query, $keyword) {
                    $sql = "centers_halls.name_ar like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('name_en', function($query, $keyword) {
                    $sql = "centers_halls.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('building_name', function($query, $keyword) {
                    $sql = "building_name like ?";
                    $query->whereRaw($sql, ["%{$keyword}%"]);
                })
            ->filterColumn('address', function($query, $keyword) {
                    $sql = "address_ar like ? or address_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('center_name', function($query, $keyword) {
                    $sql = "centers.name_ar like ? or centers.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            ->filterColumn('region_name', function($query, $keyword) {
                    $sql = "regions.name_ar like ? or regions.name_en like ?";
                    $query->whereRaw($sql, ["%{$keyword}%","%{$keyword}%"]);
                })
            // ->addColumn('image', function($row){
            //             if($row->image){
            //                 $url = url('uploads/centerHalls/'.$row->image);
            //                 return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
            //             }else{
            //                 $url = url("resources/assets/back/img/noimage.png");
            //                 return '<img src="' . $url . '" border="0" width="40" class="img-rounded" />';
            //             }
            //         })
            //  ->editColumn('status', function($row){
            //             return ($row->status == 1)?trans('home.yes'):trans('home.no');
            //  })
            ->rawColumns(['map_url','action','checkbox'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(CenterHall $model): QueryBuilder
    {
        if(auth()->user()->type=="partner"){
            $centers_ids =Partner::where('user_id',auth()->user()->id)->pluck('center_id')->toArray();
            $centers = Center::whereIn('id',$centers_ids)->get();
            return $model->with(['center','region'])
                ->leftJoin('centers','centers.id','centers_halls.center_id')
                ->leftJoin('regions','regions.id','centers_halls.region_id')
                ->select('centers_halls.*','centers.name_ar as center_name_ar','centers.name_en as center_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
                ->whereIn('centers_halls.center_id',$centers_ids)
                ->newQuery();
        }else{
            return $model->with(['center','region'])
            ->leftJoin('centers','centers.id','centers_halls.center_id')
            ->leftJoin('regions','regions.id','centers_halls.region_id')
            ->select('centers_halls.*','centers.name_ar as center_name_ar','centers.name_en as center_name_en','regions.name_ar as region_name_ar','regions.name_en as region_name_en')
            ->newQuery();
        }
        
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {   if(auth()->user()->can('table_export_buttons')) {
            if(auth()->user()->type == 'admin' || auth()->user()->type == 'super_admin') {
                $arr = ['excel','csv','print','pageLength'];
            }else if(auth()->user()->type == 'partner'){
                $arr = [];
            }else {
                $arr = ['print','pageLength'];
            }
        } else {
            $arr =[];
        }
        return $this->builder()
                    ->setTableId('centerHalls-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                     ->orderBy(1, 'asc')
                    ->selectStyleSingle()
                    ->scrollX(true)
                    ->pageLength(50)
                    ->lengthMenu([['10','25','50','100','-1'],['10','25','50','100','All']])
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'buttons'      => $arr,
                        'columnDefs' => [
                            ['targets' => [-1], 'className' => 'text-right'],
                        ],
                        'language' => [
                                'url' => '//cdn.datatables.net/plug-ins/1.13.3/i18n/ar.json',
                        ],
                        'aoColumnDefs' => [
                            // Disable sorting and searching for "details" column
                            ['aTargets' => ['checkbox'], 'bSearchable' => false, 'bSortable' => false]
                        ]
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            [
                'title' => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'class'          => 'msg_check',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            Column::make('id')
                    ->title(__('home.id')),
            Column::make('center_name')
                    ->title(__('home.partner')),
            // Column::make('name_en')
            //         ->title(__('home.hall_name_en')),
            Column::make('name_ar')
                    ->title(__('home.hall_name_ar')),
            Column::make('region_name')
                    ->title(__('home.region')),
            Column::make('building_name')
                    ->title(__('home.building_name')),
            Column::make('floor')
                    ->title(__('home.floor')),
            Column::make('address')
                    ->title(__('home.address')),
            Column::make('capacity')
                    ->title(__('home.capacity')),
             Column::make('map_url')
                    ->title(__('home.map_url')),
            // Column::make('status')
            //         ->title(__('home.status')),
            Column::make('action')
                    ->title(__('home.edit')),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'CenterHalls_' . date('YmdHis');
    }
}
